%% DICTIONARY LEARNING FOR IMAGE DENOISING
%   This file contains an example of how SMALLbox can be used to test different
%   dictionary learning techniques in Image Denoising problem.
%   It calls generateImageDenoiseProblem that will let you to choose image,
%   add noise and use noisy image to generate training set for dictionary
%   learning.
%   Three dictionary learning techniques were compared:
%   -   KSVD - M. Elad, R. Rubinstein, and M. Zibulevsky, "Efficient
%              Implementation of the K-SVD Algorithm using Batch Orthogonal
%              Matching Pursuit", Technical Report - CS, Technion, April 2008.
%   -   KSVDS - R. Rubinstein, M. Zibulevsky, and M. Elad, "Learning Sparse
%               Dictionaries for Sparse Signal Approximation", Technical
%               Report - CS, Technion, June 2009.
%   -   SPAMS - J. Mairal, F. Bach, J. Ponce and G. Sapiro. Online
%               Dictionary Learning for Sparse Coding. International
%               Conference on Machine Learning,Montreal, Canada, 2009
%
%
% Ivan Damnjanovic 2010
%%

clear;

%   If you want to load the image outside of generateImageDenoiseProblem
%   function uncomment following lines. This can be useful if you want to
%   denoise more then one image for example.

% TMPpath=pwd;
% FS=filesep;
% [pathstr1, name, ext, versn] = fileparts(which('SMALLboxSetup.m'));
% cd([pathstr1,FS,'data',FS,'images']);
% [filename,pathname] = uigetfile({'*.png;'},'Select a file containin pre-calculated notes');
% [pathstr, name, ext, versn] = fileparts(filename);
% test_image = imread(filename);
% test_image = double(test_image);
% cd(TMPpath);
% SMALL.Problem.name=name;


% Defining Image Denoising Problem as Dictionary Learning
% Problem. As an input we set the number of training patches.

SMALL.Problem = generateImageDenoiseProblem('', 40000);


%%
%   Use KSVD Dictionary Learning Algorithm to Learn overcomplete dictionary

%   Initialising Dictionary structure
%   Setting Dictionary structure fields (toolbox, name, param, D and time)
%   to zero values

SMALL.DL(1)=SMALL_init_DL();

% Defining the parameters needed for dictionary learning

SMALL.DL(1).toolbox = 'KSVD';
SMALL.DL(1).name = 'ksvd';

%   Defining the parameters for KSVD
%   In this example we are learning 256 atoms in 20 iterations, so that
%   every patch in the training set can be represented with target error in
%   L2-norm (EData)
%   Type help ksvd in MATLAB prompt for more options.

Edata=sqrt(prod(SMALL.Problem.blocksize)) * SMALL.Problem.sigma * SMALL.Problem.gain;
SMALL.DL(1).param=struct(...
    'Edata', Edata,...
    'initdict', SMALL.Problem.initdict,...
    'dictsize', SMALL.Problem.p,...
    'iternum', 20,...
    'memusage', 'high');

%   Learn the dictionary

SMALL.DL(1) = SMALL_learn(SMALL.Problem, SMALL.DL(1));

%   Set SMALL.Problem.A dictionary
%   (backward compatiblity with SPARCO: solver structure communicate
%   only with Problem structure, ie no direct communication between DL and
%   solver structures)

SMALL.Problem.A = SMALL.DL(1).D;


%%
%   Initialising solver structure
%   Setting solver structure fields (toolbox, name, param, solution,
%   reconstructed and time) to zero values

SMALL.solver(1)=SMALL_init_solver;

% Defining the parameters needed for image denoising

SMALL.solver(1).toolbox='ompbox';
SMALL.solver(1).name='ompdenoise';

%   Denoising the image - SMALL_denoise function is similar to SMALL_solve,
%   but backward compatible with KSVD definition of denoising

SMALL.solver(1)=SMALL_denoise(SMALL.Problem, SMALL.solver(1));

%%
% Use KSVDS Dictionary Learning Algorithm to denoise image

%   Initialising solver structure
%   Setting solver structure fields (toolbox, name, param, solution,
%   reconstructed and time) to zero values

SMALL.DL(2)=SMALL_init_DL();

% Defining the parameters needed for dictionary learning

SMALL.DL(2).toolbox = 'KSVDS';
SMALL.DL(2).name = 'ksvds';

%   Defining the parameters for KSVDS
%   In this example we are learning 256 atoms in 20 iterations, so that
%   every patch in the training set can be represented with target error in
%   L2-norm (EDataS). We also impose "double sparsity" - dictionary itself
%   has to be sparse in the given base dictionary (Tdict - number of
%   nonzero elements per atom).
%   Type help ksvds in MATLAB prompt for more options.

EdataS=sqrt(prod(SMALL.Problem.blocksize)) * SMALL.Problem.sigma * SMALL.Problem.gain;
SMALL.DL(2).param=struct(...
    'Edata', EdataS, ...
    'Tdict', 6,...
    'stepsize', 1,...
    'dictsize', SMALL.Problem.p,...
    'iternum', 20,...
    'memusage', 'high');
SMALL.DL(2).param.initA = speye(SMALL.Problem.p);
SMALL.DL(2).param.basedict{1} = odctdict(8,16);
SMALL.DL(2).param.basedict{2} = odctdict(8,16);

% Learn the dictionary

SMALL.DL(2) = SMALL_learn(SMALL.Problem, SMALL.DL(2));

%   Set SMALL.Problem.A dictionary and SMALL.Problem.basedictionary
%   (backward compatiblity with SPARCO: solver structure communicate
%   only with Problem structure, ie no direct communication between DL and
%   solver structures)

SMALL.Problem.A = SMALL.DL(2).D;
SMALL.Problem.basedict{1} = SMALL.DL(2).param.basedict{1};
SMALL.Problem.basedict{2} = SMALL.DL(2).param.basedict{2};

%%
%   Initialising solver structure
%   Setting solver structure fields (toolbox, name, param, solution,
%   reconstructed and time) to zero values

SMALL.solver(2)=SMALL_init_solver;

% Defining the parameters needed for image denoising

SMALL.solver(2).toolbox='ompsbox';
SMALL.solver(2).name='ompsdenoise';

%   Denoising the image - SMALL_denoise function is similar to SMALL_solve,
%   but backward compatible with KSVD definition of denoising
%   Pay attention that since implicit base dictionary is used, denoising
%   can be much faster then using explicit dictionary in KSVD example.

SMALL.solver(2)=SMALL_denoise(SMALL.Problem, SMALL.solver(2));

% %%
% %   Use SPAMS Online Dictionary Learning Algorithm
% %   to Learn overcomplete dictionary (Julien Mairal 2009)
% %   (If you have not installed SPAMS please comment the following two cells)
% 
% %   Initialising Dictionary structure
% %   Setting Dictionary structure fields (toolbox, name, param, D and time)
% %   to zero values
% 
% SMALL.DL(3)=SMALL_init_DL();
% 
% %   Defining fields needed for dictionary learning
% 
% SMALL.DL(3).toolbox = 'SPAMS';
% SMALL.DL(3).name = 'mexTrainDL';
% 
% %   Type 'help mexTrainDL in MATLAB prompt for explanation of parameters.
% 
% SMALL.DL(3).param=struct(...
%     'D', SMALL.Problem.initdict,...
%     'K', SMALL.Problem.p,...
%     'lambda', 2,...
%     'iter', 200,...
%     'mode', 3, ...
%     'modeD', 0);
% 
% %   Learn the dictionary
% 
% SMALL.DL(3) = SMALL_learn(SMALL.Problem, SMALL.DL(3));
% 
% %   Set SMALL.Problem.A dictionary
% %   (backward compatiblity with SPARCO: solver structure communicate
% %   only with Problem structure, ie no direct communication between DL and
% %   solver structures)
% 
% SMALL.Problem.A = SMALL.DL(3).D;
% 
% 
% %%
% %   Initialising solver structure
% %   Setting solver structure fields (toolbox, name, param, solution,
% %   reconstructed and time) to zero values
% 
% SMALL.solver(3)=SMALL_init_solver;
% 
% % Defining the parameters needed for denoising
% 
% SMALL.solver(3).toolbox='ompbox';
% SMALL.solver(3).name='ompdenoise';
% 
% %   Denoising the image - SMALL_denoise function is similar to SMALL_solve,
% %   but backward compatible with KSVD definition of denoising
% 
% SMALL.solver(3)=SMALL_denoise(SMALL.Problem, SMALL.solver(3));

%%
% Plot results and save midi files

% show results %

SMALL_ImgDeNoiseResult(SMALL);
